/*---------------------------------------
   PalmNotePad.C -- PalmNotePad Editor
       originally based on work by Charles Petzold
  ---------------------------------------*/


#include <windows.h>
#include <windowsx.h>

#include "resource.h"
#include <tchar.h>

#include <commctrl.h>
#include <commdlg.h>

#include <stdlib.h>

#if defined (_DEBUG)
    TCHAR   szDebug[500];
		LRESULT	iDebugRET;
#endif

#define UNTITLED TEXT("(untitled)")

//------------------------------------------------------
// Globals
//------------------------------------------------------


HINSTANCE g_hInst = NULL;  // Local copy of hInstance, used in Commandbar_Create & Commandbar_InsertMenubar
HWND g_hwndMain = NULL;    // Handle to Main window returned from CreateWindow
HWND g_hwndCB =NULL;       // Handle to a commandband
HMENU g_hMenu =NULL;       // Handle to a commandbar0 menu

HWND g_hwndCBr0 = NULL;		// Handle to the base commandbar
HWND g_hwndCBr1 = NULL;		// Handle to the first commandbar
HWND g_hwndCBr2 = NULL;		// Handle to the second commandbar

HWND g_hwndEdit = NULL;    // Handle to Main window returned from CreateWindow

UINT g_nCBHeight = 0;  //Height of the CommandBand

TCHAR g_szFileName[_MAX_PATH] = {'\0'} ;
TCHAR g_szTitleName[_MAX_FNAME + _MAX_EXT]  = {'\0'} ;

// Globals for Option Dialog state
BOOL		g_bWordWrap = FALSE;

static HWND hDlgModeless ;

static TCHAR szAppName[] = TEXT("PalmNotePad Application");
static TCHAR szTitle[]   = TEXT("PalmNotePad");

TCHAR szTextChild[] = TEXT("PalmNotePadChild");

// Array to contain the buttons for the first commandbar

static TBBUTTON tbCBr1Button[] = {
	{STD_FILENEW,    TB_STD_FILENEW, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0, 0, -1},
	{STD_FILEOPEN,   TB_STD_FILEOPEN, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0, 0, -1},
	{STD_FILESAVE,   TB_STD_FILESAVE, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0, 0, -1},
	{0,              0,        TBSTATE_ENABLED, TBSTYLE_SEP,    0, 0, 0, -1},
	{STD_CUT,        TB_STD_CUT, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0, 0,  0},
	{STD_COPY,       TB_STD_COPY, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0, 0, -1},
	{STD_PASTE,      TB_STD_PASTE, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0, 0, -1},
	{STD_UNDO,       TB_STD_UNDO, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0, 0, -1},
};

// Tooltips for the first commandbar
TCHAR * szCBr1Tips[] = {
		TEXT("New File"),
		TEXT("Open File"),
		TEXT("Save File"),
		TEXT("Cut"),
		TEXT("Copy"),
		TEXT("Paste"),
		TEXT("Undo"),
};
// Number of Tooltips on the first commandbar
UINT uCBr1TipsCount = 7;

// Array to contain the buttons for the second commandbar

static TBBUTTON tbCBr2Button[] = {
	{STD_FIND,       TB_STD_FIND, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0, 0, -1},
	{IDB_FINDNEXT,   TB_STD_FINDNEXT, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0, 0, -1},
	{STD_REPLACE,    TB_STD_REPLACE, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0, 0, -1},
	{IDB_CLEAR,				TB_STD_CLEAR, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0, 0, -1},
	{IDB_SELECTALL,   TB_STD_SELECTALL, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0, 0,  0},
	{IDB_SETFONT,     TB_STD_SETFONT, TBSTATE_ENABLED, TBSTYLE_BUTTON, 0, 0, 0, -1},
};

// Tooltips for the second commandbar
TCHAR * szCBr2Tips[] = {
		TEXT("Find"),
		TEXT("Find Next"),
		TEXT("Replace"),
		TEXT("Clear"),
		TEXT("SelectAll"),
		TEXT("Set Font"),
};
// Number of Tooltips on the second commandbar
UINT uCBr2TipsCount = 6;

static int       iOffset ;

// FIND GLOBALS

#define MAX_STRING_LEN   256

static TCHAR szFindText [MAX_STRING_LEN] ;
static TCHAR szReplText [MAX_STRING_LEN] ;

static BOOL g_bFindMatchWholeWordOnly = FALSE;
static BOOL g_bFindMatchCase = FALSE;
static BOOL g_bFindDirectionUP = FALSE; //FALSE is down



//------------------------------------------------------
// Forward Declares
//------------------------------------------------------


LRESULT CALLBACK WndProc      (HWND, UINT, WPARAM, LPARAM) ;
BOOL    CALLBACK AboutDlgProc (HWND, UINT, WPARAM, LPARAM) ;

          // Functions in PALMNOTEPADFILE.C

void PalmNotePadFileInitialize (HWND) ;
BOOL PalmNotePadFileOpenDlg    (HWND, LPTSTR, LPTSTR) ;
BOOL PalmNotePadFileSaveDlg    (HWND, LPTSTR, LPTSTR) ;
BOOL PalmNotePadFileRead       (HWND, LPTSTR) ;
BOOL PalmNotePadFileWrite      (HWND, LPTSTR) ;

          // Functions for FIND/REPLACE dialogs

BOOL PalmNotePadFindFindDlg     (HWND, HINSTANCE) ;
BOOL PalmNotePadFindReplaceDlg  (HWND, HINSTANCE) ;

BOOL PalmNotePadFindFindText    (HWND, int *, LPTSTR) ;
BOOL PalmNotePadFindReplaceText (HWND, int *, LPTSTR, LPTSTR) ;
BOOL PalmNotePadFindNextText    (HWND, int * ,LPTSTR) ;
BOOL PalmNotePadFindValidFind   (void) ;

          // Functions for FONT Dialog

void PalmNotePadFontInitialize   (HWND) ;
BOOL PalmNotePadFontDlg (HWND , HINSTANCE );
void PalmNotePadFontSetFont      (HWND) ;
void PalmNotePadFontDeinitialize (void) ;

          // Functions in POPPRNT.C

BOOL PalmNotePadPrntPrintFile (HINSTANCE, HWND, HWND, LPTSTR) ;
				

//------------------------------------------------------
// Local Functions
//------------------------------------------------------
void OkMessage (HWND hwnd, TCHAR *szMessage, TCHAR *g_szTitleName)
     {
     TCHAR szBuffer[64 + _MAX_FNAME + _MAX_EXT] ;

     wsprintf (szBuffer, szMessage, g_szTitleName[0] ? g_szTitleName : UNTITLED) ;

     MessageBox (hwnd, szBuffer, szAppName, MB_OK | MB_ICONEXCLAMATION) ;
     }

short AskAboutSave (HWND hwnd, TCHAR *g_szTitleName)
     {
     TCHAR szBuffer[64 + _MAX_FNAME + _MAX_EXT] ;
     int  iReturn ;

     wsprintf (szBuffer,TEXT("Save current changes in %s?"),g_szTitleName[0] ? g_szTitleName : UNTITLED) ;

     iReturn = MessageBox (hwnd, szBuffer, szAppName,
                           MB_YESNOCANCEL | MB_ICONQUESTION) ;

     if (iReturn == IDYES)
          if (!SendMessage (hwnd, WM_COMMAND, IDM_SAVE, 0L))
               iReturn = IDCANCEL ;

     return iReturn ;
     }

//------------------------------------------------------
//  MainWndProc
//------------------------------------------------------

LRESULT CALLBACK MainWndProc (HWND hwnd, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
	static BOOL      bNeedSave = FALSE ;
	static UINT      iMsgFindReplace ;
	int              iSelBeg, iSelEnd, iEnable ;
	RECT						 rect;

	switch (iMsg)
		{
		case WM_CREATE:
			{
			HBITMAP hbmp;

			//Array to hold toolbar info
			REBARBANDINFO arbbi[3];

			//Imagelist for custom bitmap, which is used by the RBBS_IMAGE flag
			HIMAGELIST himl = ImageList_Create(16, 16, ILC_COLOR, 2, 0);

			//Load the custom TOOLBAR pictures
			hbmp = LoadBitmap(g_hInst, MAKEINTRESOURCE(IDB_TOOLBAR_LABEL));

			//Add the TOOLBAR pictures to the image list
			ImageList_Add(himl, hbmp, 0);

			//Clean up the bitmap
			DeleteObject(hbmp);

			//Create the CommandBand to place the menu and the CommandBars on
			g_hwndCB = CommandBands_Create(g_hInst, hwnd, BANDS_ID, 
					RBS_VARHEIGHT | RBS_BANDBORDERS, himl);

			//Array values to represent the info/styles of the 3 CommandBars
			arbbi[0].cbSize = sizeof(REBARBANDINFO); 
			arbbi[0].fMask = RBBIM_ID | RBBIM_STYLE| RBBIM_SIZE;
			arbbi[0].fStyle = RBBS_NOGRIPPER ;
			arbbi[0].wID = ID_MENUBAND;
			arbbi[0].cx = 240;
			arbbi[0].iImage = 0;
			
			arbbi[1].cbSize = sizeof(REBARBANDINFO); 
			arbbi[1].fMask = RBBIM_ID |  RBBIM_IMAGE | RBBIM_SIZE | RBS_SMARTLABELS;
			arbbi[1].wID = ID_BUTTONBAND0;
			arbbi[1].cx = 240;
			arbbi[1].iImage = 0;

			arbbi[2].cbSize = sizeof(REBARBANDINFO); 
			arbbi[2].fMask = RBBIM_ID | RBBIM_IMAGE | RBBIM_SIZE | RBS_SMARTLABELS;
			arbbi[2].wID = ID_BUTTONBAND1;
			arbbi[2].cx = 240;
			arbbi[2].iImage = 1;

			//Add the 3 CommandBars we just defined to the CommandBand
			CommandBands_AddBands(g_hwndCB, g_hInst, 3, arbbi);

			//Get the hwnd of the first commandbar
			g_hwndCBr0 = CommandBands_GetCommandBar(g_hwndCB, 0);

			//Put the menu on the first commandbar
			CommandBar_InsertMenubar(g_hwndCBr0, g_hInst, IDR_MAINMENU, 0);

			//Get the menu handle (in case we may need it later
			g_hMenu = CommandBar_GetMenu(g_hwndCBr0,0);

			//Get the Second Commandbar
			g_hwndCBr1 = CommandBands_GetCommandBar(g_hwndCB, 1);

			//Add the Bitmaps to the Second Commandbar
			//These are system standard bitmaps
			CommandBar_AddBitmap(g_hwndCBr1, HINST_COMMCTRL, IDB_STD_SMALL_COLOR, 15, 16, 16);
			
			//Add the Buttons to the Second Commandbar
			//And stick the Bitmaps (as defined in the tbCBr1Button array, look up above) on the buttons
			CommandBar_AddButtons(g_hwndCBr1, sizeof(tbCBr1Button)/sizeof(TBBUTTON), tbCBr1Button);

			//Add the tooltips to the buttons (as defined in the szCBr1Tips array, look up above)
			CommandBar_AddToolTips(g_hwndCBr1, uCBr1TipsCount, szCBr1Tips);

			//Get the handle to the third Commandbar
			g_hwndCBr2 = CommandBands_GetCommandBar(g_hwndCB, 2);

			//AddBitmaps for the third Commandbar (standard bitmaps)
			CommandBar_AddBitmap(g_hwndCBr2, HINST_COMMCTRL, IDB_STD_SMALL_COLOR, 15, 16, 16);
			
			//Add Custom bitmaps for the third Commandbar
			CommandBar_AddBitmap(g_hwndCBr2, g_hInst, IDB_TOOLBAR_BITMAP, 7, 16, 16);

			//Add the buttons and past the bitmaps to them
			CommandBar_AddButtons(g_hwndCBr2, sizeof(tbCBr2Button)/sizeof(TBBUTTON), tbCBr2Button);

			//Add the third commandbar bitmaps
			CommandBar_AddToolTips(g_hwndCBr2, uCBr2TipsCount, szCBr2Tips);

			//Get the Client rectangle size
			GetClientRect(hwnd, &rect);
	
			//Get and store the CommandBand height, because the commandBand is part of the client
			//window space
			g_nCBHeight = CommandBands_Height(g_hwndCB);

			//Create the Edit Control
			g_hwndEdit = CreateWindow(TEXT("edit"),
									NULL,
                  WS_CHILD | WS_VISIBLE | WS_HSCROLL | WS_VSCROLL |
                  WS_BORDER | ES_LEFT | ES_MULTILINE |
                  ES_NOHIDESEL, 
									0, 
									g_nCBHeight, 
									rect.right-rect.left, 
									rect.bottom- (rect.top+g_nCBHeight), 
									hwnd,
									(HMENU)EDIT_ID, 
									g_hInst,
									NULL);

			SendMessage (g_hwndEdit, EM_SETLIMITTEXT, (WPARAM)0, 0L) ;
			if ( g_hwndEdit )
			{
				ShowWindow(g_hwndEdit, SW_SHOW);
				UpdateWindow(g_hwndEdit);
			}

			// Initialize common dialog box stuff

			PalmNotePadFileInitialize (hwnd) ;
			PalmNotePadFontInitialize (g_hwndEdit) ;

			if (lstrlen (g_szFileName) > 0)
			{
				if (!PalmNotePadFileRead (g_hwndEdit, g_szFileName))
				{
					 OkMessage (hwnd, TEXT("File %s cannot be read!"),g_szFileName) ;
				}
			}
		}
   return 0 ;

		case WM_SETFOCUS :
		 SetFocus (g_hwndEdit) ;
		 return 0 ;


		case WM_INITMENUPOPUP :
			switch (lParam)
			{
				case 1 :        // Edit menu

						// Enable Undo if edit control can do it

					EnableMenuItem ((HMENU) wParam, IDM_UNDO,
						SendMessage (g_hwndEdit, EM_CANUNDO, 0, 0L) ?
								 MF_ENABLED : MF_GRAYED) ;

						// Enable Paste if text is in the clipboard

					EnableMenuItem ((HMENU) wParam, IDM_PASTE,
						IsClipboardFormatAvailable (CF_TEXT) ?
								 MF_ENABLED : MF_GRAYED) ;

						// Enable Cut, Copy, and Del if text is selected

					SendMessage (g_hwndEdit, EM_GETSEL, (WPARAM) &iSelBeg,
																				 (LPARAM) &iSelEnd) ;

					iEnable = iSelBeg != iSelEnd ? MF_ENABLED : MF_GRAYED ;

					EnableMenuItem ((HMENU) wParam, IDM_CUT,   iEnable) ;
					EnableMenuItem ((HMENU) wParam, IDM_COPY,  iEnable) ;
					EnableMenuItem ((HMENU) wParam, IDM_CLEAR, iEnable) ;
					break ;

				case 2 :        // Search menu

							// Enable Find, Next, and Replace if modeless
							//   dialogs are not already active

					iEnable = hDlgModeless == NULL ?
												MF_ENABLED : MF_GRAYED ;

					EnableMenuItem ((HMENU) wParam, IDM_FIND,    iEnable) ;
					EnableMenuItem ((HMENU) wParam, IDM_NEXT,    iEnable) ;
					EnableMenuItem ((HMENU) wParam, IDM_REPLACE, iEnable) ;
					break ;
			}
			return 0 ;

			case WM_NOTIFY:
				{			
					if (wParam == BANDS_ID)
					{
						case RBN_LAYOUTCHANGED:
							GetClientRect(hwnd, &rect);
				
							if (g_nCBHeight != CommandBands_Height(g_hwndCB))
							{
								g_nCBHeight = CommandBands_Height(g_hwndCB);
								MoveWindow(g_hwndEdit, 0, g_nCBHeight, 
												rect.right-rect.left, 
												rect.bottom- (rect.top+g_nCBHeight),
												TRUE);
							};
							return DefWindowProc(hwnd, iMsg, wParam, lParam);
					}
				}
				return DefWindowProc(hwnd, iMsg, wParam, lParam);

			case WM_COMMAND :
			// Messages from edit control

			 if (lParam && LOWORD (wParam) == EDIT_ID)
				{
				switch (HIWORD (wParam))
						 {
						 case EN_UPDATE :
									bNeedSave = TRUE ;
									return 0 ;

						 case EN_ERRSPACE :
						 case EN_MAXTEXT :
									MessageBox (hwnd, TEXT("Edit control out of space."),szAppName, MB_OK | MB_ICONSTOP) ;
									return 0 ;
						 }
				break ;
				}

			switch (LOWORD (wParam))
			{
				// Messages from File menu
				// And the CommandBar buttons as designated by the TB_* notation
				case IDM_EDIT_WORDWRAP :
					if (!g_bWordWrap) 
					{
						CheckMenuItem(g_hMenu, IDM_EDIT_WORDWRAP, MF_CHECKED );
						g_bWordWrap = TRUE;
						DestroyWindow(g_hwndEdit);
						GetClientRect(g_hwndMain, &rect);
						g_hwndEdit = CreateWindow(TEXT("edit"),
												NULL,
												WS_CHILD | WS_VISIBLE | WS_VSCROLL |
												WS_BORDER | ES_LEFT | ES_MULTILINE |
												ES_NOHIDESEL, 
												0, 
												g_nCBHeight, 
												rect.right-rect.left, 
												rect.bottom- (rect.top+g_nCBHeight), 
												g_hwndMain,
												(HMENU)EDIT_ID, 
												g_hInst,
												NULL);

						SendMessage (g_hwndEdit, EM_SETLIMITTEXT, (WPARAM)0, 0L) ;
						if ( g_hwndEdit )
						{
							ShowWindow(g_hwndEdit, SW_SHOW);
							UpdateWindow(g_hwndEdit);
						}
						PalmNotePadFileRead (g_hwndEdit, g_szFileName);
					} else
					{
						CheckMenuItem(g_hMenu, IDM_EDIT_WORDWRAP, MF_UNCHECKED );
						g_bWordWrap = FALSE;
						DestroyWindow(g_hwndEdit);
						GetClientRect(g_hwndMain, &rect);
						g_hwndEdit = CreateWindow(TEXT("edit"),
												NULL,
												WS_CHILD | WS_VISIBLE | WS_VSCROLL | WS_HSCROLL |
												WS_BORDER | ES_LEFT | ES_MULTILINE |
												ES_NOHIDESEL, 
												0, 
												g_nCBHeight, 
												rect.right-rect.left, 
												rect.bottom- (rect.top+g_nCBHeight), 
												g_hwndMain,
												(HMENU)EDIT_ID, 
												g_hInst,
												NULL);

						SendMessage (g_hwndEdit, EM_SETLIMITTEXT, (WPARAM)0, 0L) ;
						if ( g_hwndEdit )
						{
							ShowWindow(g_hwndEdit, SW_SHOW);
							UpdateWindow(g_hwndEdit);
						}
						PalmNotePadFileRead (g_hwndEdit, g_szFileName);
					}
					return 0;


				case TB_STD_FILENEW :
				case IDM_NEW :
					if (bNeedSave && IDCANCEL ==
									 AskAboutSave (hwnd, g_szTitleName))
							return 0 ;

					SetWindowText (g_hwndEdit, TEXT("\0")) ;
					g_szFileName[0]  = '\0' ;
					g_szTitleName[0] = '\0' ;
					bNeedSave = FALSE ;
					return 0 ;

				case TB_STD_FILEOPEN :
				case IDM_OPEN :
				 if (bNeedSave && IDCANCEL ==
									 AskAboutSave (hwnd, g_szTitleName))
							return 0 ;

				 if (PalmNotePadFileOpenDlg (hwnd, g_szFileName, g_szTitleName))
							{
							if (!PalmNotePadFileRead (g_hwndEdit, g_szFileName))
									 {
									 OkMessage (hwnd, TEXT("Could not read file %s!"),g_szTitleName) ;
									 g_szFileName[0]  = '\0' ;
									 g_szTitleName[0] = '\0' ;
									 }
							}

				 bNeedSave = FALSE ;
				 return 0 ;

				case TB_STD_FILESAVE :
				case IDM_SAVE :
					if (g_szFileName[0])
					{
						if (PalmNotePadFileWrite (g_hwndEdit, g_szFileName))
						{
							bNeedSave = FALSE ;
							return 1 ;
						}
						else
							OkMessage (hwnd, TEXT("Could not write file %s"), g_szTitleName) ;
						return 0 ;
					}
																		// fall through
				case IDM_SAVEAS :
					if (PalmNotePadFileSaveDlg (hwnd, g_szFileName, g_szTitleName))
					{

					if (PalmNotePadFileWrite (g_hwndEdit, g_szFileName))
					{
						bNeedSave = FALSE ;
						return 1 ;
					}
					else
						OkMessage (hwnd, TEXT("Could not write file %s"), g_szTitleName) ;
					}
					return 0 ;

				case IDM_PRINT :
				 if (!PalmNotePadPrntPrintFile (g_hInst, hwnd, g_hwndEdit,g_szTitleName))
							OkMessage (hwnd, TEXT("Could not print file %s"),g_szTitleName) ;
				 return 0 ;

				case IDM_EXIT :
				 SendMessage (hwnd, WM_CLOSE, 0, 0) ;
				 return 0 ;

								// Messages from Edit menu

				case TB_STD_UNDO :
				case IDM_UNDO :
				 SendMessage (g_hwndEdit, WM_UNDO, 0, 0) ;
				 return 0 ;

				case TB_STD_CUT :
				case IDM_CUT :
				 SendMessage (g_hwndEdit, WM_CUT, 0, 0) ;
				 return 0 ;

				case TB_STD_COPY :
				case IDM_COPY :
				 SendMessage (g_hwndEdit, WM_COPY, 0, 0) ;
				 return 0 ;

				case TB_STD_PASTE :
				case IDM_PASTE :
				 SendMessage (g_hwndEdit, WM_PASTE, 0, 0) ;
				 return 0 ;

				case TB_STD_CLEAR :
				case IDM_CLEAR :
				 SendMessage (g_hwndEdit, WM_CLEAR, 0, 0) ;
				 return 0 ;

				case TB_STD_SELECTALL :
				case IDM_SELALL :
				 SendMessage (g_hwndEdit, EM_SETSEL, 0, -1) ;
				 return 0 ;

								// Messages from Search menu

				case TB_STD_FIND :
				case IDM_FIND :
				 SendMessage (g_hwndEdit, EM_GETSEL, 0,(LPARAM) &iOffset) ;

				 PalmNotePadFindFindDlg (hwnd, g_hInst) ;
				 return 0 ;

				case TB_STD_FINDNEXT :
				case IDM_NEXT :
				 SendMessage (g_hwndEdit, EM_GETSEL, 0,(LPARAM) &iOffset) ;

				 if (PalmNotePadFindValidFind ())
				      PalmNotePadFindNextText (g_hwndEdit, &iOffset, szFindText) ;
				 else
				      PalmNotePadFindFindDlg (hwnd, g_hInst) ;

				 return 0 ;

				case TB_STD_REPLACE :
				case IDM_REPLACE :
				 SendMessage (g_hwndEdit, EM_GETSEL, 0, (LPARAM) &iOffset) ;

				 PalmNotePadFindReplaceDlg (hwnd, g_hInst) ;
				 return 0 ;

				case TB_STD_SETFONT :
				case IDM_FONT :
				 if (PalmNotePadFontDlg (hwnd, g_hInst))
				      PalmNotePadFontSetFont (g_hwndEdit) ;

				 return 0 ;

				// Messages from Help menu

				case IDM_HELP :
						 OkMessage (hwnd, TEXT("Help not yet implemented!"), TEXT("\0")) ;
						 return 0 ;

				case IDM_ABOUT :
						 DialogBox (g_hInst, TEXT("AboutBox"), hwnd, AboutDlgProc) ;
						 return 0 ;
				}
				break ; //WM_COMMAND


			case WM_CLOSE :
				if (!bNeedSave || IDCANCEL != AskAboutSave (hwnd, g_szTitleName))
						DestroyWindow (hwnd) ;

				return 0 ;

			case WM_DESTROY :
				PalmNotePadFontDeinitialize () ;
				PostQuitMessage (0) ;
				return 0 ;

			default:
                         
		break ;
		}
	return DefWindowProc (hwnd, iMsg, wParam, lParam) ;
}

//------------------------------------------------------
//  AboutDlgProc
//------------------------------------------------------

BOOL CALLBACK AboutDlgProc (HWND hDlg, UINT iMsg, WPARAM wParam, LPARAM lParam)
{
	switch (iMsg)
	{
		case WM_INITDIALOG :
		 return TRUE ;

		case WM_COMMAND :
		 switch (LOWORD (wParam))
			{
			case IDOK :
			 EndDialog (hDlg, 0) ;
			 return TRUE ;
			}
		 break ;
	}
	return FALSE ;
}

//------------------------------------------------------
//  InitInstance
//------------------------------------------------------


BOOL InitInstance (HINSTANCE hInstance, int CmdShow )
{

	g_hInst = hInstance;

	g_hwndMain = CreateWindow(szAppName,						
													szTitle,                        
													WS_VISIBLE,
													0,
													0,
													CW_USEDEFAULT,
													CW_USEDEFAULT,
													NULL, 
													NULL, 
													hInstance, 
													NULL );

	if ( !g_hwndMain )
		
	{
		return FALSE;
	}

	ShowWindow(g_hwndMain, CmdShow );

	UpdateWindow(g_hwndMain);

	return TRUE;
}

//------------------------------------------------------
//   InitApplication
//------------------------------------------------------
BOOL InitApplication ( HINSTANCE hInstance )
{
	WNDCLASS wc;
	BOOL f;

	//Commoncontrol structure
	//This is required to use CommandBands
	INITCOMMONCONTROLSEX icex;

	wc.style = CS_HREDRAW | CS_VREDRAW ;
	wc.lpfnWndProc = (WNDPROC)MainWndProc;
	wc.cbClsExtra = 0;
	wc.cbWndExtra = 0;
	wc.hIcon = LoadIcon(hInstance, MAKEINTRESOURCE(IDI_ICON1));//NULL;
	wc.hInstance = hInstance;
	wc.hCursor = NULL;                                             
	wc.hbrBackground = (HBRUSH) GetStockObject( WHITE_BRUSH );
	wc.lpszMenuName = NULL;
	wc.lpszClassName = szAppName;
	
	f = (RegisterClass(&wc));


	//This is the initialization for the CommandBands structure
	icex.dwSize = sizeof(icex);
	icex.dwICC = ICC_BAR_CLASSES | ICC_COOL_CLASSES;
	InitCommonControlsEx(&icex);

  return f;
}


//------------------------------------------------------
//  WinMain
//------------------------------------------------------

int WINAPI WinMain(HINSTANCE hInstance,
                   HINSTANCE hPrevInstance,
                   LPTSTR     lpCmdLine,
									 int        CmdShow)

{

	MSG msg;
	HWND hPalmNotePadwnd = NULL;

	//Get commandline file if it exists
	if (lpCmdLine[0] != '\0')
		{
		_tcscpy (g_szFileName, (LPTSTR)lpCmdLine);
		}

	//Get name of file to write out for changes
	if (lpCmdLine[0] != '\0')
		{
		_tcscpy (g_szTitleName, _tcsrchr((LPTSTR)lpCmdLine, '\\') + 1);
		}

	//Check to see if app is running 
	//If it is then pop PalmNotePad to foreground

	hPalmNotePadwnd = FindWindow(szAppName, szTitle);	
	if (hPalmNotePadwnd) 
	{
		SetForegroundWindow (hPalmNotePadwnd);    
		return 0;
	}

	if ( !hPrevInstance )
	{
		if ( !InitApplication ( hInstance ) )
		{ 
			return (FALSE); 
		}

	}

	if ( !InitInstance( hInstance, CmdShow )  )
		return (FALSE);

    while ( GetMessage( &msg, NULL, 0,0 ) )
	{
		TranslateMessage (&msg);
		DispatchMessage(&msg);
	}

	return (msg.wParam);
}


//------------------------------------------------------
// FindDlgProc(HWND, WORD, WORD, LONG)
//------------------------------------------------------

BOOL CALLBACK FindDlgProc(HWND hwnd, WORD msg, WORD wp, LONG lp)

{  
	LRESULT	lResult = TRUE;

	switch(msg)
	{
		case WM_COMMAND:
			switch (GET_WM_COMMAND_ID(wp,lp))
				{
				case IDB_FIND_FINDNEXT:
					//get find text 
					Edit_GetText(GetDlgItem(hwnd,IDE_FIND_FINDWHAT), szFindText, MAX_STRING_LEN); 
					//Do find
					if (!PalmNotePadFindFindText (g_hwndEdit, &iOffset, szFindText)) OkMessage (hwnd, TEXT("Text not found!"), TEXT("\0")) ;
					break;

				case IDB_FIND_CANCEL:
					EndDialog(hwnd, TRUE);
					break;
				}
			break;

			case WM_INITDIALOG:
				//set find text 
				if(szFindText) Edit_SetText(GetDlgItem(hwnd,IDE_FIND_FINDWHAT), szFindText); 
				
				break;

		default:
			return(FALSE);
			break;
	}

  UNREFERENCED_PARAMETER(wp);
  UNREFERENCED_PARAMETER(lp);
	return (lResult);
}


//------------------------------------------------------
// ReplaceDlgProc(HWND, WORD, WORD, LONG)
//------------------------------------------------------

BOOL CALLBACK ReplaceDlgProc(HWND hwnd, WORD msg, WORD wp, LONG lp)

{  
	LRESULT	lResult = TRUE;

	switch(msg)
	{
		case WM_COMMAND:
			switch (GET_WM_COMMAND_ID(wp,lp))
				{
				case IDB_REPLACE_FINDNEXT:
					//get find text 
					Edit_GetText(GetDlgItem(hwnd,IDE_REPLACE_FINDWHAT), szFindText, MAX_STRING_LEN);
					Edit_GetText(GetDlgItem(hwnd,IDE_REPLACE_REPLACEWITH), szReplText, MAX_STRING_LEN);
					//Do find
					if (!PalmNotePadFindFindText (g_hwndEdit, &iOffset, szFindText)) OkMessage (hwnd, TEXT("Text not found!"), TEXT("\0")) ;
					break;

				case IDB_REPLACE_REPLACE:
					//get dialog field values
					//get find text 
					Edit_GetText(GetDlgItem(hwnd,IDE_REPLACE_FINDWHAT), szFindText, MAX_STRING_LEN);
					Edit_GetText(GetDlgItem(hwnd,IDE_REPLACE_REPLACEWITH), szReplText, MAX_STRING_LEN);
					//Do find
					if (!PalmNotePadFindReplaceText (g_hwndEdit, &iOffset, szFindText, szReplText)) OkMessage (hwnd, TEXT("Text not found!"), TEXT("\0")) ;
					break;

				case IDB_REPLACE_REPLACEALL:
					//get dialog field values
					//get find text 
					Edit_GetText(GetDlgItem(hwnd,IDE_REPLACE_FINDWHAT), szFindText, MAX_STRING_LEN);
					Edit_GetText(GetDlgItem(hwnd,IDE_REPLACE_REPLACEWITH), szReplText, MAX_STRING_LEN);

					//Do replace
					iOffset = 0;

					while ((PalmNotePadFindReplaceText (g_hwndEdit, &iOffset, szFindText, szReplText)) == TRUE);
					
					break;

				case IDB_REPLACE_CANCEL:
					EndDialog(hwnd, TRUE);
					break;
				}
			break;

			case WM_INITDIALOG:
				//set find text 
				if(szFindText) Edit_SetText(GetDlgItem(hwnd,IDE_REPLACE_FINDWHAT), szFindText); 
				if(szReplText) Edit_SetText(GetDlgItem(hwnd,IDE_REPLACE_REPLACEWITH), szReplText);
				break;

		default:
			return(FALSE);
			break;
	}

  UNREFERENCED_PARAMETER(wp);
  UNREFERENCED_PARAMETER(lp);
	return (lResult);
}
//------------------------------------------------------
// PalmNotePadFindFindDlg
//------------------------------------------------------

BOOL PalmNotePadFindFindDlg (HWND hwnd, HINSTANCE g_hInst)
{
	return(DialogBox (g_hInst, MAKEINTRESOURCE(IDD_FIND), hwnd, (DLGPROC)FindDlgProc)) ;
}

//------------------------------------------------------
// PalmNotePadFindFindDlg
//------------------------------------------------------

BOOL PalmNotePadFindReplaceDlg (HWND hwnd, HINSTANCE g_hInst)
{
	return(DialogBox (g_hInst, MAKEINTRESOURCE(IDD_REPLACE), hwnd, (DLGPROC)ReplaceDlgProc)) ;

}

//------------------------------------------------------
// PalmNotePadFindFindText
//------------------------------------------------------


BOOL PalmNotePadFindFindText (HWND hwndEdit, int *piSearchOffset, LPTSTR szFindText)
{
	int   iLength, iPos ;
	LPTSTR  lptstrDoc, lptstrPos ;

					 // Read in the edit document

	iLength = GetWindowTextLength (hwndEdit) ;

	if (NULL == (lptstrDoc = (LPTSTR) malloc ((iLength + 1) * sizeof(TCHAR))))
			return FALSE ;

	GetWindowText (hwndEdit, lptstrDoc, iLength + 1) ;

					 // Search the document for the find string
	lptstrPos = _tcsstr (lptstrDoc + *piSearchOffset, szFindText) ;
	free (lptstrDoc) ;

					 // Return an error code if the string cannot be found

	if (lptstrPos == NULL)
			return FALSE ;

					 // Find the position in the document and the new start offset

	iPos = lptstrPos - lptstrDoc ;
	*piSearchOffset = iPos + lstrlen (szFindText) ;

					 // Select the found text

	SendMessage (hwndEdit, EM_SETSEL, iPos, *piSearchOffset) ;
	SendMessage (hwndEdit, EM_SCROLLCARET, 0, 0) ;

	return TRUE ;
}

//------------------------------------------------------
// PalmNotePadFindNextText
//------------------------------------------------------

BOOL PalmNotePadFindNextText (HWND hwndEdit, int *piSearchOffset, LPTSTR szFindText)
{
	return PalmNotePadFindFindText (hwndEdit, piSearchOffset, szFindText) ;
}


//------------------------------------------------------
// PalmNotePadFindReplaceText
//------------------------------------------------------

BOOL PalmNotePadFindReplaceText (HWND hwndEdit, int *piSearchOffset, LPTSTR szFindText, LPTSTR szReplText)
{
					 // Find the text

	if (!PalmNotePadFindFindText (hwndEdit, piSearchOffset, szFindText))
			return FALSE ;

					 // Replace it

	SendMessage (hwndEdit, EM_REPLACESEL, 0, (LPARAM) szReplText) ;

	return TRUE ;
}

//------------------------------------------------------
// PalmNotePadFindValidFind
//------------------------------------------------------

BOOL PalmNotePadFindValidFind (void)
{
	return *szFindText != '\0' ;
}

/*------------------------------------------
   PalmNotePadFONT  Popup Editor Font Functions
  ------------------------------------------*/

static LOGFONT logfont ;
static HFONT   hFont ;

//------------------------------------------------------
// FontDlgProc(HWND, WORD, WORD, LONG)
//------------------------------------------------------

BOOL CALLBACK FontDlgProc(HWND hwnd, WORD msg, WORD wp, LONG lp)

{  
	LRESULT	lResult = TRUE;

	switch(msg)
	{
		case WM_COMMAND:
			switch (GET_WM_COMMAND_ID(wp,lp))
				{
				case IDB_FONT_APPLY:
					//WM_SETFONT
					break;

				case IDB_FONT_OK:
					//WM_SETFONT
					EndDialog(hwnd, TRUE);
					break;
				
				case IDB_REPLACE_CANCEL:
					EndDialog(hwnd, TRUE);
					break;
				}
			break;

			case WM_INITDIALOG:
				//initialize the font ComboBoxes here.
				break;

		default:
			return(FALSE);
			break;
	}

  UNREFERENCED_PARAMETER(wp);
  UNREFERENCED_PARAMETER(lp);
	return (lResult);
}
//------------------------------------------------------
// PalmNotePadFontDlg
//------------------------------------------------------

BOOL PalmNotePadFontDlg (HWND hwnd, HINSTANCE hInst)
{
	return(DialogBox (hInst, MAKEINTRESOURCE(IDD_FONT), hwnd, (DLGPROC)FontDlgProc)) ;
}

//------------------------------------------------------
// PalmNotePadFontInitialize
//------------------------------------------------------

void PalmNotePadFontInitialize (HWND hwndEdit)
{
	GetObject (GetStockObject (SYSTEM_FONT), sizeof (LOGFONT),
																					(PSTR) &logfont) ;
	hFont = CreateFontIndirect (&logfont) ;
	SendMessage (hwndEdit, WM_SETFONT, (WPARAM) hFont, 0) ;
}

//------------------------------------------------------
// PalmNotePadFontSetFont
//------------------------------------------------------

void PalmNotePadFontSetFont (HWND hwndEdit)
{
	HFONT hFontNew ;
	RECT  rect ;

	hFontNew = CreateFontIndirect (&logfont) ;
	SendMessage (hwndEdit, WM_SETFONT, (WPARAM) hFontNew, 0) ;
	DeleteObject (hFont) ;
	hFont = hFontNew ;
	GetClientRect (hwndEdit, &rect) ;
	InvalidateRect (hwndEdit, &rect, TRUE) ;
}

//------------------------------------------------------
// PalmNotePadFontDeinitialize
//------------------------------------------------------

void PalmNotePadFontDeinitialize (void)
{
	DeleteObject (hFont) ;
}


